/*:ja
 * @target MZ
 * @plugindesc 会話中に立ち絵を表示・管理するプラグイン
 * @author 柊　七紀
 * @help
 * このプラグインは、メッセージ表示中にキャラクターの立ち絵を表示・管理します。
 * 立ち絵画像は「img/pictures/立ち絵/キャラクター名/」フォルダに配置してください。
 * ベース画像は「キャラクター名.png」、表情画像は「表情名.png」です。
 *
 * @param ---Default Settings---
 * @text ---デフォルト設定---
 *
 * @param DefaultPositionXLeft
 * @parent ---Default Settings---
 * @type number
 * @min -9999
 * @max 9999
 * @default 0
 * @text 左立ち絵X座標オフセット
 * @desc 左側の立ち絵のX座標の調整値です。0で画面幅の1/4の位置。
 *
 * @param DefaultPositionXRight
 * @parent ---Default Settings---
 * @type number
 * @min -9999
 * @max 9999
 * @default 0
 * @text 右立ち絵X座標オフセット
 * @desc 右側の立ち絵のX座標の調整値です。0で画面幅の3/4の位置。
 *
 * @param DefaultPositionY
 * @parent ---Default Settings---
 * @type number
 * @min -9999
 * @max 9999
 * @default 0
 * @text 立ち絵Y座標オフセット
 * @desc 立ち絵のY座標の調整値です。0で画面下端に足元が来ます。
 *
 * @param DefaultScaleX
 * @parent ---Default Settings---
 * @type number
 * @min 0.01
 * @max 10.0
 * @decimals 2
 * @default 1.00
 * @text 立ち絵横スケール
 * @desc 立ち絵の横方向の拡大率です。(1.0 = 100%)
 *
 * @param DefaultScaleY
 * @parent ---Default Settings---
 * @type number
 * @min 0.01
 * @max 10.0
 * @decimals 2
 * @default 1.00
 * @text 立ち絵縦スケール
 * @desc 立ち絵の縦方向の拡大率です。(1.0 = 100%)
 *
 * @param DefaultCostumeVariableId
 * @parent ---Default Settings---
 * @type variable
 * @default 0
 * @text デフォルト衣装変数ID
 * @desc この変数に設定された文字列が、指定キャラクターのデフォルト衣装として適用されます。空文字列で衣装なし。
 *
 * @param TargetCharaForDefaultCostume
 * @parent ---Default Settings---
 * @type string
 * @default シトラス
 * @text デフォルト衣装適用キャラクター
 * @desc デフォルト衣装変数(DefaultCostumeVariableId)を適用するキャラクターの名前。
 *
 * @command ShowStandingPicture
 * @text 立ち絵表示/変更
 * @desc 指定したキャラクターの立ち絵と表情を表示、または変更します。
 * @arg argsString
 * @text 引数文字列
 * @desc 例: キャラクター名・表情名・位置 (例: シトラス・通常・左・衣装)
 * 位置を省略すると「左」になります。
 * 表情名のみ指定すると、直近の立ち絵の表情を変更します (例: 笑顔)
 * 衣装名を省略すると、指定キャラクターの場合のみデフォルト衣装変数の値が使用されます。
 * 衣装名を「なし」と指定すると、変数にかかわらず衣装は表示されません。
 * 区切り文字は「・」（全角中点）です。
 *
 * @command HideStandingPicture
 * @text 立ち絵非表示
 * @desc 指定した位置の立ち絵を非表示にします。引数なしで全て非表示。
 * @arg position
 * @text 位置
 * @desc 'left' または 'right'。省略すると左右両方を非表示にします。
 * @type select
 * @option (全て)
 * @value all
 * @option 左
 * @value left
 * @option 右
 * @value right
 * @default all
 */

(() => {
    const PLUGIN_NAME = "CoF_Message";
    const BASE_PICTURE_PATH = '立ち絵/'; // img/pictures/ のサブフォルダ名

    // プラグインパラメータの取得
    const pluginParams = PluginManager.parameters(PLUGIN_NAME);
    const defaultPositionXLeft = Number(pluginParams.DefaultPositionXLeft || 0);
    const defaultPositionXRight = Number(pluginParams.DefaultPositionXRight || 0);
    const defaultPositionY = Number(pluginParams.DefaultPositionY || 0);
    const defaultScaleX = Number(pluginParams.DefaultScaleX || 1.0);
    const defaultScaleY = Number(pluginParams.DefaultScaleY || 1.0);
    const defaultCostumeVariableId = Number(pluginParams.DefaultCostumeVariableId || 0);
    const TARGET_CHARA_FOR_DEFAULT_COSTUME = pluginParams.TargetCharaForDefaultCostume || ''; // デフォルト衣装変数を適用するキャラクター名

    // 立ち絵スプライトと最終表示情報を管理するオブジェクト
    const _Game_Temp_initialize = Game_Temp.prototype.initialize;
    Game_Temp.prototype.initialize = function() {
        _Game_Temp_initialize.call(this);
        this._standingPictures = {
            left: null, // 左側の立ち絵情報 {bodySprite, faceSprite, charaName, faceName, costumeName}
            right: null // 右側の立ち絵情報
        };
        // 最後に表示された立ち絵の情報 (表情変更の簡略化用)
        // ロードに成功した画像の情報のみを保持する
        this._lastDisplayedStandingPicture = {
            charaName: null,
            position: null, // 'left' or 'right'
            faceName: null,   // 成功した表情名
            costumeName: null // 成功した衣装名
        };
        this._messageWindowHidden = false; // 会話ウィンドウの非表示状態 (完全に非表示用)
        this._messageWindowPaused = false; // 会話ウィンドウの進行停止状態
    };

    /**
     * 指定された画像パスのBitmapがロード完了するまで待機する
     * @param {string} path ImageManager.loadPictureに渡すパス (例: '立ち絵/シトラス/シトラス')
     * @returns {Promise<boolean>} ロードが完了したらtrue、失敗したらfalseを解決するPromise
     */
    const waitForImageLoad = (path) => {
        return new Promise(resolve => {
            const bitmap = ImageManager.loadPicture(path); // 画像をロードキューに登録
            if (bitmap.isReady()) {
                // すでにロード済みの場合は即座に解決
                resolve(true);
                return;
            }
            // ロード完了またはエラーを待機
            const checkInterval = setInterval(() => {
                if (bitmap.isReady()) {
                    clearInterval(checkInterval);
                    resolve(true);
                } else if (bitmap.isError()) {
                    clearInterval(checkInterval);
                    // ここでImageManagerからのFailed to loadエラーは防げないが、
                    // プラグインが独自に検知したエラーとしてログに出す
                    console.error(`${PLUGIN_NAME}: 画像のロードに失敗、または見つかりません: img/pictures/${path}.png`);
                    resolve(false);
                }
            }, 10); // 10msごとにチェック
        });
    };

    /**
     * 現在の衣装名を決定するヘルパー関数
     * @param {string} charaName キャラクター名
     * @param {string|null} argCostumeName コマンドで指定された衣装名（省略された場合はnull）
     * @returns {string|null} 実際に使用する衣装名、またはnull（衣装なしの場合）
     */
    const resolveCostumeName = (charaName, argCostumeName) => {
        // 明示的に「なし」または空文字列が指定された場合は衣装なし
        if (argCostumeName === 'なし' || argCostumeName === '') {
            return null;
        } 
        // コマンドで具体的に指定された衣装名がある場合
        else if (argCostumeName) {
            return argCostumeName;
        } 
        // 指定キャラクターに限り、デフォルト衣装変数を参照
        else if (charaName === TARGET_CHARA_FOR_DEFAULT_COSTUME && defaultCostumeVariableId > 0) {
            const varValue = $gameVariables.value(defaultCostumeVariableId);
            if (typeof varValue === 'string' && varValue !== '') {
                return varValue; // 変数に有効な衣装名が設定されている場合
            }
        }
        return null; // それ以外の場合は衣装なし
    };

    //=============================================================================
    // Plugin Commands
    //=============================================================================

    // 立ち絵表示/変更コマンド
    PluginManager.registerCommand(PLUGIN_NAME, "ShowStandingPicture", async args => {
        const argsString = args.argsString; // 例: シトラス・通常・左・制服
        const params = argsString.split('・'); // 全角中点で分割

        let charaName;
        let faceName;
        let position;
        let specifiedCostumeName = null; // コマンドで直接指定された衣装名
        let resolvedCostumeName = null;  // 最終的に使用する衣装名

        let bodyImagePath = null; // ベースまたは衣装画像のパス
        let faceImagePath = null;

        if (params.length === 1) {
            // 表情名のみの場合 (例: 笑顔)
            faceName = params[0];
            const lastPic = $gameTemp._lastDisplayedStandingPicture;

            if (!lastPic.charaName || !lastPic.position) {
                console.warn(`${PLUGIN_NAME}: 表情名のみの指定ですが、以前に表示された立ち絵情報が見つかりません。キャラクター名と位置も指定してください。`, argsString);
                return;
            }
            charaName = lastPic.charaName;
            position = lastPic.position;
            // 表情変更時も、以前の衣装を維持しつつデフォルト衣装変数を考慮
            resolvedCostumeName = resolveCostumeName(charaName, lastPic.costumeName); 

            // 衣装があれば衣装画像をボディとして使用、なければ通常ベース
            bodyImagePath = resolvedCostumeName ? `${BASE_PICTURE_PATH}${charaName}/${resolvedCostumeName}` : `${BASE_PICTURE_PATH}${charaName}/${charaName}`;
            faceImagePath = `${BASE_PICTURE_PATH}${charaName}/${faceName}`;

            // 全ての画像のロード完了を待機
            const loadPromises = [
                waitForImageLoad(bodyImagePath),
                waitForImageLoad(faceImagePath)
            ];
            const results = await Promise.all(loadPromises);

            // ロードが成功した場合のみ立ち絵を更新
            if (results[0] && results[1]) {
                const scene = SceneManager._scene;
                if (scene instanceof Scene_Map || scene instanceof Scene_Battle) {
                    scene.createOrUpdateStandingPicture(charaName, faceName, position, resolvedCostumeName);
                    // 最後に表示した立ち絵情報を更新
                    $gameTemp._lastDisplayedStandingPicture.charaName = charaName;
                    $gameTemp._lastDisplayedStandingPicture.position = position;
                    $gameTemp._lastDisplayedStandingPicture.faceName = faceName; // 成功した表情名
                    $gameTemp._lastDisplayedStandingPicture.costumeName = resolvedCostumeName;
                } else {
                    console.warn(`${PLUGIN_NAME}: 'ShowStandingPicture' はマップまたは戦闘シーンでのみ使用できます。`);
                }
            } else {
                // ロード失敗時でも、すでにImageManagerがエラーを吐いているはずなので、
                // ここでのwarnは開発者向けの情報として残す
                console.warn(`${PLUGIN_NAME}: 立ち絵の表情変更に失敗しました。画像がロードできませんでした。指定: ${argsString}`);
            }

        } else if (params.length >= 2) {
            // フル指定または部分指定の場合
            charaName = params[0];
            faceName = params[1];
            position = 'left'; // 位置指定がない場合はデフォルトで左

            if (params.length >= 3) {
                const pos = params[2].toLowerCase();
                if (pos === '左' || pos === 'left') {
                    position = 'left';
                } else if (pos === '右' || pos === 'right') {
                    position = 'right';
                } else {
                    // 位置指定が不正な場合は、3番目の引数を衣装名として解釈し、位置はデフォルトの左とする
                    specifiedCostumeName = params[2];
                }
            }
            
            // 4つ目の引数がある場合は衣装名（3つ目の引数が位置でなかった場合も含む）
            if (params.length >= 4) {
                specifiedCostumeName = params[3];
            }

            // charaNameと指定された衣装名で最終的な衣装名を解決
            resolvedCostumeName = resolveCostumeName(charaName, specifiedCostumeName); 

            // 衣装があれば衣装画像をボディとして使用、なければ通常ベース
            bodyImagePath = resolvedCostumeName ? `${BASE_PICTURE_PATH}${charaName}/${resolvedCostumeName}` : `${BASE_PICTURE_PATH}${charaName}/${charaName}`;
            faceImagePath = `${BASE_PICTURE_PATH}${charaName}/${faceName}`;

            // 全ての画像のロード完了を待機
            const loadPromises = [
                waitForImageLoad(bodyImagePath),
                waitForImageLoad(faceImagePath)
            ];

            const results = await Promise.all(loadPromises);

            // ロードが成功した場合のみ立ち絵を更新
            if (results[0] && results[1]) {
                const scene = SceneManager._scene;
                if (scene instanceof Scene_Map || scene instanceof Scene_Battle) {
                    scene.createOrUpdateStandingPicture(charaName, faceName, position, resolvedCostumeName);
                    // 最後に表示した立ち絵情報を更新
                    $gameTemp._lastDisplayedStandingPicture.charaName = charaName;
                    $gameTemp._lastDisplayedStandingPicture.position = position;
                    $gameTemp._lastDisplayedStandingPicture.faceName = faceName; // 成功した表情名
                    $gameTemp._lastDisplayedStandingPicture.costumeName = resolvedCostumeName;
                } else {
                    console.warn(`${PLUGIN_NAME}: 'ShowStandingPicture' はマップまたは戦闘シーンでのみ使用できます。`);
                }
            } else {
                // ロード失敗時でも、すでにImageManagerがエラーを吐いているはずなので、
                // ここでのwarnは開発者向けの情報として残す
                console.warn(`${PLUGIN_NAME}: 立ち絵の表示に失敗しました。画像がロードできませんでした。指定: ${argsString}`);
            }

        } else {
            console.error(`${PLUGIN_NAME}: 'ShowStandingPicture' コマンドの引数が不正です。`, argsString);
            return;
        }
    });

    // 立ち絵非表示コマンド
    PluginManager.registerCommand(PLUGIN_NAME, "HideStandingPicture", args => {
        const position = args.position; // 'all', 'left', 'right'

        const scene = SceneManager._scene;
        if (scene instanceof Scene_Map || scene instanceof Scene_Battle) {
            if (position === 'all') {
                scene.clearStandingPicture('left');
                scene.clearStandingPicture('right');
            } else {
                scene.clearStandingPicture(position);
            }
            // 立ち絵を非表示にした場合、最後に表示した立ち絵情報をクリアする
            $gameTemp._lastDisplayedStandingPicture.charaName = null;
            $gameTemp._lastDisplayedStandingPicture.position = null;
            $gameTemp._lastDisplayedStandingPicture.faceName = null; // クリア
            $gameTemp._lastDisplayedStandingPicture.costumeName = null;
        } else {
            console.warn(`${PLUGIN_NAME}: 'HideStandingPicture' はマップまたは戦闘シーンでのみ使用できます。`);
        }
    });


    //=============================================================================
    // Scene_Base への機能追加
    // (Scene_MapとScene_Battleで共通して立ち絵を扱うため)
    //=============================================================================

    /**
     * 立ち絵の初期設定と表示。この関数はShowStandingPictureコマンドからのみ呼び出すことを想定。
     * 呼び出し元で画像のロード成功が確認されている必要がある。
     */
    Scene_Base.prototype.createOrUpdateStandingPicture = function(charaName, faceName, position, costumeName = null) {
        const faceImagePath = `${BASE_PICTURE_PATH}${charaName}/${faceName}`;
        
        let bodyBitmap; // ベースまたは衣装のBitmap
        if (costumeName) {
            const costumeImagePath = `${BASE_PICTURE_PATH}${charaName}/${costumeName}`;
            bodyBitmap = ImageManager.loadPicture(costumeImagePath);
        } else {
            const baseImagePath = `${BASE_PICTURE_PATH}${charaName}/${charaName}`;
            bodyBitmap = ImageManager.loadPicture(baseImagePath);
        }
        
        const faceBitmap = ImageManager.loadPicture(faceImagePath);

        // スプライト作成直前で、両方のBitmapがロード完了していることを確認
        // ロードに失敗したBitmapをスプライトに設定すると描画エラーや透明化の原因となるため、ここで完全にブロックする
        if (!bodyBitmap.isReady() || !faceBitmap.isReady()) {
            console.error(`${PLUGIN_NAME}: 立ち絵の作成/更新をスキップ。画像がロードできませんでした。` +
                          `キャラクター: ${charaName}, 表情: ${faceName}` +
                          (costumeName ? `, 衣装: ${costumeName}` : ''));
            // スプライト作成をスキップし、既存の立ち絵があればそれを維持する
            // (既に透明化している場合は、透明のままとなる)
            return;
        }

        // スプライトのZ座標 (メッセージウィンドウより奥)
        const Z_INDEX_BODY = -1; // 衣装がベースになるので、衣装/ベース画像はZ:-1
        const Z_INDEX_FACE = 0; // 表情はZ:0

        let standingPic = $gameTemp._standingPictures[position];

        // 既存のスプライトがある場合、一旦削除
        if (standingPic) {
            if (standingPic.bodySprite && this._spriteset && this._spriteset._pictureContainer) {
                this._spriteset._pictureContainer.removeChild(standingPic.bodySprite);
                standingPic.bodySprite.destroy();
            }
            if (standingPic.faceSprite && this._spriteset && this._spriteset._pictureContainer) {
                this._spriteset._pictureContainer.removeChild(standingPic.faceSprite);
                standingPic.faceSprite.destroy();
            }
        }
        
        standingPic = { // 新しいスプライト情報を初期化
            bodySprite: null, // ベースまたは衣装のスプライト
            faceSprite: null,
            charaName: charaName, // 現在表示しているキャラクター名を保存
            faceName: faceName,   // 現在表示している表情名を保存
            costumeName: costumeName // 現在表示している衣装名を保存
        };
        $gameTemp._standingPictures[position] = standingPic; // Game_Tempに情報を保存

        // スプライトの作成と共通設定
        const setupSprite = (bitmap, zIndex) => {
            const sprite = new Sprite(bitmap);
            sprite.anchor.x = 0.5;
            sprite.anchor.y = 1;
            sprite.zIndex = zIndex;
            sprite.scale.x = defaultScaleX; // プラグインパラメータからスケールを適用
            sprite.scale.y = defaultScaleY; // プラグインパラメータからスケールを適用
            return sprite;
        };

        const bodySprite = setupSprite(bodyBitmap, Z_INDEX_BODY); // ベースまたは衣装
        const faceSprite = setupSprite(faceBitmap, Z_INDEX_FACE);
        
        // 位置調整
        const screenWidth = Graphics.boxWidth;
        const screenHeight = Graphics.boxHeight;
        
        let targetX;
        if (position === 'left') {
            targetX = screenWidth / 4 + defaultPositionXLeft; // 左用オフセットを適用
        } else { // right
            targetX = screenWidth * 3 / 4 + defaultPositionXRight; // 右用オフsetを適用
        }
        const targetY = screenHeight + defaultPositionY; // Y用オフセットを適用

        bodySprite.x = targetX;
        bodySprite.y = targetY;
        
        // 表情スプライトはベース（または衣装）スプライトと同じ位置に
        faceSprite.x = targetX;
        faceSprite.y = targetY;
        
        // PictureContainer に追加（MZの推奨）
        if (this._spriteset && this._spriteset._pictureContainer) {
            this._spriteset._pictureContainer.addChild(bodySprite);
            this._spriteset._pictureContainer.addChild(faceSprite);
            // Zオーダーを再ソート
            this._spriteset._pictureContainer.children.sort((a, b) => (a.zIndex || 0) - (b.zIndex || 0));
        } else {
            console.warn(`${PLUGIN_NAME}: スプライトセットまたはピクチャーコンテナが見つかりません。立ち絵の表示に問題がある可能性があります。`);
            this.addChild(bodySprite);
            this.addChild(faceSprite);
            this.children.sort((a, b) => (a.zIndex || 0) - (b.zIndex || 0));
        }

        // スプライトオブジェクトをGame_Tempに保存
        standingPic.bodySprite = bodySprite;
        standingPic.faceSprite = faceSprite;

        console.log(`${PLUGIN_NAME}: 立ち絵表示 - ${charaName} (${faceName}) @ ${position}` + (costumeName ? ` with ${costumeName}` : ''));
    };

    // 立ち絵を非表示にする
    Scene_Base.prototype.clearStandingPicture = function(position) {
        const standingPic = $gameTemp._standingPictures[position];
        if (standingPic) {
            if (standingPic.bodySprite && this._spriteset && this._spriteset._pictureContainer) {
                this._spriteset._pictureContainer.removeChild(standingPic.bodySprite);
                standingPic.bodySprite.destroy();
            }
            if (standingPic.faceSprite && this._spriteset && this._spriteset._pictureContainer) {
                this._spriteset._pictureContainer.removeChild(standingPic.faceSprite);
                standingPic.faceSprite.destroy();
            }
            // 該当位置の立ち絵情報をクリア
            $gameTemp._standingPictures[position] = null; 
            console.log(`${PLUGIN_NAME}: 立ち絵非表示 - ${position}`);
        }
    };
    
    // シーン切り替え時に立ち絵をクリア (マップから戦闘へ、またはその逆)
    const _Scene_Map_terminate = Scene_Map.prototype.terminate;
    Scene_Map.prototype.terminate = function() {
        _Scene_Map_terminate.call(this);
        // マップ終了時に立ち絵をクリア
        this.clearStandingPicture('left');
        this.clearStandingPicture('right');
        // _lastDisplayedStandingPicture もクリア
        $gameTemp._lastDisplayedStandingPicture.charaName = null;
        $gameTemp._lastDisplayedStandingPicture.position = null;
        $gameTemp._lastDisplayedStandingPicture.faceName = null; // クリア
        $gameTemp._lastDisplayedStandingPicture.costumeName = null;

        // シーン終了時にメッセージウィンドウ非表示状態もリセット
        $gameTemp._messageWindowHidden = false;
        $gameTemp._messageWindowPaused = false;
    };

    const _Scene_Battle_terminate = Scene_Battle.prototype.terminate;
    Scene_Battle.prototype.terminate = function() {
        _Scene_Battle_terminate.call(this);
        // 戦闘終了時に立ち絵をクリア
        this.clearStandingPicture('left');
        this.clearStandingPicture('right');
        // _lastDisplayedStandingPicture もクリア
        $gameTemp._lastDisplayedStandingPicture.charaName = null;
        $gameTemp._lastDisplayedStandingPicture.position = null;
        $gameTemp._lastDisplayedStandingPicture.faceName = null; // クリア
        $gameTemp._lastDisplayedStandingPicture.costumeName = null;

        // シーン終了時にメッセージウィンドウ非表示状態もリセット
        $gameTemp._messageWindowHidden = false;
        $gameTemp._messageWindowPaused = false;
    };
    
    // イベントコマンド実行時に立ち絵を更新または再表示
    const _Window_Message_startMessage = Window_Message.prototype.startMessage;
    Window_Message.prototype.startMessage = function() {
        _Window_Message_startMessage.call(this);
        const scene = SceneManager._scene;
        if (scene instanceof Scene_Map || scene instanceof Scene_Battle) {
            const standingPics = $gameTemp._standingPictures;
            const promises = [];

            // 左側の立ち絵の再描画をリクエスト
            if (standingPics.left && standingPics.left.charaName && standingPics.left.faceName) {
                const charaName = standingPics.left.charaName;
                const faceName = standingPics.left.faceName;
                const costumeName = standingPics.left.costumeName; // 既に解決済みの衣装名を使用

                promises.push(new Promise(async resolve => {
                    const bodyImagePath = costumeName ? `${BASE_PICTURE_PATH}${charaName}/${costumeName}` : `${BASE_PICTURE_PATH}${charaName}/${charaName}`;
                    const faceImagePath = `${BASE_PICTURE_PATH}${charaName}/${faceName}`;

                    const bodyLoaded = await waitForImageLoad(bodyImagePath);
                    const faceLoaded = await waitForImageLoad(faceImagePath);
                    
                    if (bodyLoaded && faceLoaded) {
                        scene.createOrUpdateStandingPicture(charaName, faceName, 'left', costumeName);
                    } else {
                        console.warn(`${PLUGIN_NAME}: 左立ち絵の再描画に必要な画像の一部がロードできませんでした。透明化を防ぐため表示をクリアします。`);
                        scene.clearStandingPicture('left'); // 明示的にクリアして透明化を回避
                    }
                    resolve();
                }));
            }

            // 右側の立ち絵の再描画をリクエスト
            if (standingPics.right && standingPics.right.charaName && standingPics.right.faceName) {
                const charaName = standingPics.right.charaName;
                const faceName = standingPics.right.faceName;
                const costumeName = standingPics.right.costumeName; // 既に解決済みの衣装名を使用

                promises.push(new Promise(async resolve => {
                    const bodyImagePath = costumeName ? `${BASE_PICTURE_PATH}${charaName}/${costumeName}` : `${BASE_PICTURE_PATH}${charaName}/${charaName}`;
                    const faceImagePath = `${BASE_PICTURE_PATH}${charaName}/${faceName}`;

                    const bodyLoaded = await waitForImageLoad(bodyImagePath);
                    const faceLoaded = await waitForImageLoad(faceImagePath);

                    if (bodyLoaded && faceLoaded) {
                        scene.createOrUpdateStandingPicture(charaName, faceName, 'right', costumeName);
                    } else {
                        console.warn(`${PLUGIN_NAME}: 右立ち絵の再描画に必要な画像の一部がロードできませんでした。透明化を防ぐため表示をクリアします。`);
                        scene.clearStandingPicture('right'); // 明示的にクリアして透明化を回避
                    }
                    resolve();
                }));
            }

            // すべての再描画が完了するのを待つ (ただし、イベントフローはブロックしない)
            Promise.all(promises).catch(error => {
                console.error(`${PLUGIN_NAME}: 立ち絵の再描画中にエラーが発生しました:`, error);
            });
        }
    };

    //=============================================================================
    // 会話ウィンドウの非表示機能の追加
    //=============================================================================

    // Game_Tempに会話ウィンドウの非表示状態を保持するプロパティを追加済み
    // (this._messageWindowHidden = false; this._messageWindowPaused = false;)

    const _Scene_Map_update = Scene_Map.prototype.update;
    Scene_Map.prototype.update = function() {
        _Scene_Map_update.call(this);
        // 会話中かつメッセージウィンドウがアクティブな場合のみ処理
        if ($gameMessage.isBusy() && this._messageWindow && this._messageWindow.isOpen()) {
            this.handleMessageWindowVisibility();
        }
    };

    const _Scene_Battle_update = Scene_Battle.prototype.update;
    Scene_Battle.prototype.update = function() {
        _Scene_Battle_update.call(this);
        // 会話中かつメッセージウィンドウがアクティブな場合のみ処理
        if ($gameMessage.isBusy() && this._messageWindow && this._messageWindow.isOpen()) {
            this.handleMessageWindowVisibility();
        }
    };

    // メッセージウィンドウの表示/非表示を切り替える関数
    Scene_Base.prototype.handleMessageWindowVisibility = function() {
        // キーボード: Shiftキー ('shift') または Aキー ('ok' - デフォルトの決定キーですが、メッセージ中は競合しにくい)
        // マウス: 右クリック ('cancel')
        // Input.isTriggeredは一度しか反応しないため、長押しで繰り返し切り替わらない
        // Shiftキー: 'shift'
        // Ctrlキー: 'control'
        // Altキー: 'control' (MZでは'control'にマップされている場合があります)
        // Aキー: 'ok' または 'pageup' / 'pagedown' などを試す
        // 他のキーを使いたい場合、Input.keyMapperを参照してください
        // 例: F1キーは90番なので Input.isTriggered(90)

        // ここではShiftキーと右クリックを推奨します
        if (Input.isTriggered('shift') || TouchInput.isCancelled()) {
            $gameTemp._messageWindowHidden = !$gameTemp._messageWindowHidden; // 表示/非表示状態を反転
            $gameTemp._messageWindowPaused = $gameTemp._messageWindowHidden; // 非表示ならポーズ、表示ならポーズ解除

            this.updateMessageWindowVisibility(); // 表示を更新
        }
    };

    // メッセージウィンドウの表示状態とポーズ状態を更新する関数
    Scene_Base.prototype.updateMessageWindowVisibility = function() {
        if (this._messageWindow) {
            // ウィンドウ全体の可視性を制御
            this._messageWindow.visible = !$gameTemp._messageWindowHidden;

            // 非表示状態ならメッセージ進行を停止 (ポーズ)
            // この_opennessは開閉アニメーションに影響するので直接触らない
            this._messageWindow.pause = $gameTemp._messageWindowPaused;
            
            // 名前ウィンドウも同様にvisibleを制御
            if (this._messageWindow._nameBoxWindow) {
                this._messageWindow._nameBoxWindow.visible = !$gameTemp._messageWindowHidden;
            }
            // 選択肢や数値入力などのウィンドウは、非表示中も表示されるべきではないため、
            // それらが開いている場合はメッセージウィンドウは非表示にならないようにする
            // (通常はメッセージウィンドウが非表示になったらこれらのウィンドウも閉じられるか、
            // visible=falseに設定されるはずなので、このロジックは主にメッセージウィンドウ自体に適用)
        }
        // 選択肢ウィンドウなどが開いている場合は、メッセージウィンドウは強制的に表示状態に戻す
        // これがないと、メッセージ非表示中に選択肢が開いた場合、選択肢が見えないままになる
        if (this._messageWindow && (this._messageWindow._choiceListWindow.active ||
                                     this._messageWindow._numberInputWindow.active ||
                                     this._messageWindow._eventItemWindow.active)) {
            $gameTemp._messageWindowHidden = false;
            $gameTemp._messageWindowPaused = false;
            this._messageWindow.visible = true;
            this._messageWindow.pause = false;
            if (this._messageWindow._nameBoxWindow) {
                this._messageWindow._nameBoxWindow.visible = true;
            }
        }
    };

    // メッセージウィンドウが開く際に、非表示状態をリセットする
    // これにより、新しいメッセージが表示されたときは常に表示状態から始まる
    const _Window_Message_open = Window_Message.prototype.open;
    Window_Message.prototype.open = function() {
        _Window_Message_open.call(this);
        // 新しいメッセージ表示時には、非表示フラグとポーズフラグをリセットして表示状態にする
        if ($gameTemp._messageWindowHidden || $gameTemp._messageWindowPaused) {
            $gameTemp._messageWindowHidden = false;
            $gameTemp._messageWindowPaused = false;
            // シーンのupdateMessageWindowVisibilityを呼び出して即座に反映
            if (SceneManager._scene instanceof Scene_Map || SceneManager._scene instanceof Scene_Battle) {
                SceneManager._scene.updateMessageWindowVisibility();
            }
        }
    };

    // メッセージウィンドウが閉じるときも、非表示状態をリセットする
    const _Window_Message_close = Window_Message.prototype.close;
    Window_Message.prototype.close = function() {
        _Window_Message_close.call(this);
        // メッセージが完全に閉じられたら、非表示フラグとポーズフラグをリセット
        if ($gameTemp._messageWindowHidden || $gameTemp._messageWindowPaused) {
            $gameTemp._messageWindowHidden = false;
            $gameTemp._messageWindowPaused = false;
            // シーンのupdateMessageWindowVisibilityを呼び出して即座に反映
            if (SceneManager._scene instanceof Scene_Map || SceneManager._scene instanceof Scene_Battle) {
                SceneManager._scene.updateMessageWindowVisibility();
            }
        }
    };

    // 選択肢や数値入力などが表示された時もウィンドウの透明度を更新
    // これらのウィンドウが開いている間はメッセージウィンドウは非表示にならないようにする
    const _Window_ChoiceList_open = Window_ChoiceList.prototype.open;
    Window_ChoiceList.prototype.open = function() {
        _Window_ChoiceList_open.call(this);
        if (SceneManager._scene instanceof Scene_Map || SceneManager._scene instanceof Scene_Battle) {
            SceneManager._scene.updateMessageWindowVisibility();
        }
    };

    const _Window_NumberInput_open = Window_NumberInput.prototype.open;
    Window_NumberInput.prototype.open = function() {
        _Window_NumberInput_open.call(this);
        if (SceneManager._scene instanceof Scene_Map || SceneManager._scene instanceof Scene_Battle) {
            SceneManager._scene.updateMessageWindowVisibility();
        }
    };

    const _Window_EventItem_open = Window_EventItem.prototype.open;
    Window_EventItem.prototype.open = function() {
        _Window_EventItem_open.call(this);
        if (SceneManager._scene instanceof Scene_Map || SceneManager._scene instanceof Scene_Battle) {
            SceneManager._scene.updateMessageWindowVisibility();
        }
    };

})();